// === Tabs Filter ===
// Switch between event categories using tab buttons
const tabButtons = document.querySelectorAll('.tab-button');
const gallerySections = document.querySelectorAll('.gallery-section');

tabButtons.forEach(button => {
  button.addEventListener('click', () => {
    // Remove active class from all buttons
    tabButtons.forEach(btn => btn.classList.remove('active'));
    button.classList.add('active');

    // Show matching section, hide others
    const category = button.getAttribute('data-category');
    gallerySections.forEach(section => {
      section.classList.toggle('hidden', section.getAttribute('data-category') !== category);
    });
  });
});

// === Load More Button ===
// Reveal all hidden sections or collapse back to selected tab
const loadMoreBtn = document.getElementById('loadMoreBtn');
let loaded = false;

loadMoreBtn.addEventListener('click', () => {
  if (!loaded) {
    document.querySelectorAll('.gallery-section.hidden').forEach(section => {
      section.classList.remove('hidden');
    });
    loadMoreBtn.textContent = 'Show Less';
    loaded = true;
  } else {
    const activeCategory = document.querySelector('.tab-button.active').getAttribute('data-category');
    gallerySections.forEach(section => {
      section.classList.toggle('hidden', section.getAttribute('data-category') !== activeCategory);
    });
    loadMoreBtn.textContent = 'Load More';
    loaded = false;
  }
});

// === Lightbox Viewer with Section-Specific Slideshow ===
const lightbox = document.getElementById('lightbox');
const lightboxImg = document.getElementById('lightboxImage');
const closeLightbox = document.querySelector('.close-lightbox');
const prevBtn = document.getElementById('prevBtn');
const nextBtn = document.getElementById('nextBtn');

let currentIndex = 0;
let currentGroup = []; // holds images/videos of the active section

// Open lightbox scoped to section
document.querySelectorAll('.gallery-section').forEach(section => {
  const mediaItems = section.querySelectorAll('.lightbox-trigger');
  
  mediaItems.forEach((item, index) => {
    item.addEventListener('click', () => {
      currentGroup = Array.from(mediaItems); // only this section
      currentIndex = index;
      showImage(currentIndex);
      lightbox.style.display = 'block';
    });
  });
});

// Show image function
function showImage(index) {
  lightboxImg.src = currentGroup[index].src;
}

// Next/Prev buttons
nextBtn.addEventListener('click', (e) => {
  e.stopPropagation();
  currentIndex = (currentIndex + 1) % currentGroup.length;
  showImage(currentIndex);
});

prevBtn.addEventListener('click', (e) => {
  e.stopPropagation();
  currentIndex = (currentIndex - 1 + currentGroup.length) % currentGroup.length;
  showImage(currentIndex);
});

// Keyboard navigation
document.addEventListener('keydown', (e) => {
  if (lightbox.style.display === 'block') {
    if (e.key === 'ArrowRight') {
      currentIndex = (currentIndex + 1) % currentGroup.length;
      showImage(currentIndex);
    } else if (e.key === 'ArrowLeft') {
      currentIndex = (currentIndex - 1 + currentGroup.length) % currentGroup.length;
      showImage(currentIndex);
    } else if (e.key === 'Escape') {
      lightbox.style.display = 'none';
    }
  }
});

// Close lightbox
closeLightbox.addEventListener('click', (e) => {
  e.stopPropagation();
  lightbox.style.display = 'none';
});

// Close when clicking outside image
lightbox.addEventListener('click', (e) => {
  if (e.target === lightbox) {
    lightbox.style.display = 'none';
  }
});

// === Auto-play YouTube preview on hover ===
document.querySelectorAll('.video-preview').forEach(preview => {
  const videoId = preview.getAttribute('data-video');

  preview.addEventListener('mouseenter', () => {
    preview.innerHTML = `
      <iframe src="https://www.youtube.com/embed/${videoId}?autoplay=1&mute=1&controls=0&showinfo=0&rel=0" allow="autoplay" frameborder="0" allowfullscreen></iframe>
    `;
  });

  preview.addEventListener('mouseleave', () => {
    preview.innerHTML = '';
  });
});

// === Sponsor Modal ===
const sponsorBtn = document.getElementById('sponsorBtn');
const sponsorModal = document.getElementById('sponsorModal');
const closeModal = document.querySelector('.close-modal');

sponsorBtn.addEventListener('click', () => {
  sponsorModal.style.display = 'block';
});

closeModal.addEventListener('click', () => {
  sponsorModal.style.display = 'none';
});

window.addEventListener('click', (e) => {
  if (e.target === sponsorModal) {
    sponsorModal.style.display = 'none';
  }
});