// === GLOBALS ===
let selectedCurrency = "USD";
let selectedAmountUSD = 50;
let isMonthly = false;

// === LIVE EXCHANGE RATES ===
const exchangeRates = {
  USD: 1,
  TZS: 2430,
  EUR: 0.8578,
  GBP: 0.7438
};

const currencySymbols = {
  USD: "$",
  TZS: "TSh",
  EUR: "€",
  GBP: "£"
};

// === UPDATE AMOUNT BUTTONS ===
function updateAmountButtons() {
  document.querySelectorAll(".amount-btn").forEach(btn => {
    const usd = parseInt(btn.dataset.usd);
    const converted = Math.round(usd * exchangeRates[selectedCurrency]);
    btn.textContent = `${currencySymbols[selectedCurrency]}${converted}`;
  });
}

// === SELECT AMOUNT BUTTON ===
function selectAmount(usd) {
  selectedAmountUSD = usd;
  const converted = Math.round(usd * exchangeRates[selectedCurrency]);
  document.getElementById("custom-amount").value = converted;
  document.querySelectorAll(".amount-btn").forEach(btn => btn.classList.remove("active"));
  document.querySelector(`.amount-btn[data-usd="${usd}"]`).classList.add("active");
  updateTotalDisplay();
}

// === CURRENCY CHANGE ===
document.getElementById("currency-select").addEventListener("change", e => {
  selectedCurrency = e.target.value;
  updateAmountButtons();

  // Convert placeholder value if custom amount is entered
  const input = document.getElementById("custom-amount");
  const currentValue = parseFloat(input.value);
  if (!isNaN(currentValue)) {
    const converted = Math.round(currentValue / exchangeRates[selectedCurrency] * exchangeRates[e.target.value]);
    input.value = converted;
    selectedAmountUSD = Math.round(converted / exchangeRates[selectedCurrency]);
  }

  updateTotalDisplay();
});

// === AMOUNT BUTTON CLICK ===
document.querySelectorAll(".amount-btn").forEach(btn => {
  btn.addEventListener("click", () => {
    selectAmount(parseInt(btn.dataset.usd));
  });
});

// === TYPE BUTTON CLICK ===
document.querySelectorAll(".type-btn").forEach(btn => {
  btn.addEventListener("click", () => {
    document.querySelectorAll(".type-btn").forEach(b => b.classList.remove("active"));
    btn.classList.add("active");
    isMonthly = btn.dataset.type === "monthly";
  });
});

// === STEP NAVIGATION ===
function goToStep(step) {
  for (let i = 1; i <= 3; i++) {
    document.getElementById("step-" + i).style.display = i === step ? "block" : "none";
  }
  document.getElementById("step-indicator").textContent = `Step ${step} of 3`;
  updateTotalDisplay();
}

// === TOTAL DISPLAY ===
function updateTotalDisplay() {
  const converted = Math.round(selectedAmountUSD * exchangeRates[selectedCurrency]);
  document.getElementById("total-value").textContent = `${currencySymbols[selectedCurrency]}${converted}`;
  document.getElementById("donate-button").textContent = `Donate ${currencySymbols[selectedCurrency]}${converted}`;
}

// === VALIDATE CUSTOM AMOUNT INPUT ===
document.getElementById("custom-amount").addEventListener("input", e => {
  const value = e.target.value;
  const warning = document.getElementById("amount-warning");

  if (/[^0-9]/.test(value)) {
    warning.style.display = "block";
    e.target.value = value.replace(/[^0-9]/g, "");
  } else {
    warning.style.display = "none";
    selectedAmountUSD = Math.round(parseFloat(value) / exchangeRates[selectedCurrency]);
    updateTotalDisplay();
  }
});

// === STEP 1 NEXT BUTTON VALIDATION ===
document.getElementById("step1-next").addEventListener("click", () => {
  const amount = document.getElementById("custom-amount").value;
  if (!amount || isNaN(amount) || parseFloat(amount) <= 0) {
    alert("Please enter a valid donation amount.");
    return;
  }
  goToStep(2);
});

// === STRIPE SETUP ===
const stripe = Stripe("pk_test_YourPublishableKeyHere");
const elements = stripe.elements();
const card = elements.create("card");
card.mount("#card-element");

card.on("change", event => {
  document.getElementById("card-errors").textContent = event.error ? event.error.message : "";
});

// === COUNTRY DROPDOWN ===
document.addEventListener("DOMContentLoaded", () => {
  const select = document.getElementById("country-select");
  const countries = Intl.supportedValuesOf('region').map(code => new Intl.DisplayNames(['en'], {type: 'region'}).of(code));
  countries.forEach(country => {
    const option = document.createElement("option");
    option.textContent = country;
    select.appendChild(option);
  });
  updateAmountButtons();
  updateTotalDisplay();
});

// === FINAL SUBMISSION ===
document.getElementById("donate-button").addEventListener("click", async (e) => {
  e.preventDefault();

  const amount = Math.round(selectedAmountUSD * exchangeRates[selectedCurrency]);
  const currency = selectedCurrency;
  const email = document.getElementById("email").value;
  const name = document.getElementById("first-name").value + " " + document.getElementById("last-name").value;
  const message = document.getElementById("message").value;
  const anonymous = document.getElementById("anonymous-checkbox").checked;
  const country = document.getElementById("country-select").value;

  const res = await fetch("donation_process.php", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ amount, currency, email, name, message, anonymous, country, isMonthly })
  });

  const data = await res.json();
  if (!data.success) {
    alert("Error: " + data.error);
    return;
  }

  const result = await stripe.confirmCardPayment(data.client_secret, {
    payment_method: {
      card: card,
      billing_details: { name, email }
    }
  });

  if (result.error) {
    document.getElementById("card-errors").textContent = result.error.message;
  } else if (result.paymentIntent.status === "succeeded") {
    alert("Thank you! Your donation was successful.");
    window.location.href = "support_us.html";
  }
});