<?php
session_start();
require 'db_config.php';

$error = "";

// Initialize attempt tracking
if (!isset($_SESSION['login_attempts'])) {
  $_SESSION['login_attempts'] = 0;
  $_SESSION['first_attempt_time'] = time();
}

// Check if user is locked out
$lockoutDuration = 600; // 10 minutes
$maxAttempts = 5;

if ($_SESSION['login_attempts'] >= $maxAttempts) {
  $elapsed = time() - $_SESSION['first_attempt_time'];
  if ($elapsed < $lockoutDuration) {
    $remaining = ceil(($lockoutDuration - $elapsed) / 60);
    $error = "Too many failed attempts. Please try again in $remaining minute(s).";
  } else {
    $_SESSION['login_attempts'] = 0;
    $_SESSION['first_attempt_time'] = time();
  }
}

// Proceed with login if not locked
if ($_SERVER["REQUEST_METHOD"] === "POST" && $_SESSION['login_attempts'] < $maxAttempts) {
  $email = trim($_POST['email']);
  $password = $_POST['password'];

  if (!$email || !$password) {
    $error = "Please enter both email and password.";
  } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $error = "Invalid email format.";
  } else {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = :email");
    $stmt->execute(['email' => $email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($user && password_verify($password, $user['password'])) {
      $_SESSION['login_attempts'] = 0;
      $_SESSION['first_attempt_time'] = time();

      $_SESSION['user'] = [
        'name' => $user['name'],
        'email' => $user['email'],
        'role' => $user['role']
      ];

      $update = $pdo->prepare("UPDATE users SET last_active = NOW() WHERE email = :email");
      $update->execute(['email' => $email]);

      $role = strtolower(trim($user['role']));
      if ($role === 'admin') {
        header("Location: admin_choice.php");
        exit;
      } elseif ($role === 'donor') {
        header("Location: donor_dashboard.php");
        exit;
      } else {
        $error = "Unknown user role.";
      }
    } else {
      $_SESSION['login_attempts']++;
      if ($_SESSION['login_attempts'] === 1) {
        $_SESSION['first_attempt_time'] = time();
      }
      $error = "Invalid email or password.";
    }
  }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Login | TRD</title>
  <link rel="stylesheet" href="style/style.css" />
  <link rel="stylesheet" href="style/lore.css" />
</head>
<body>
  <header class="header">
    <div class="header-top">
      <img src="assets/icons/trd logo transparent logo.png" alt="TRD Logo" class="logo-transparent" />
      <h1 class="org-name">Tanzania Rugby Development</h1>
    </div>
    <nav class="nav_links" id="navLinks">
      <a href="index.html" class="btn btn-light">Home</a>
      <a href="about.html">About Us</a>
    </nav>
  </header>

  <div class="container">
    <h3 class="mb-4 text-center">Login</h3>

    <?php if (isset($_GET['registered'])): ?>
      <p style="color:green">Registration successful. Please log in to continue.</p>
    <?php endif; ?>

    <?php if ($error): ?>
      <p style="color:red"><?= $error ?></p>
    <?php endif; ?>

    <form action="login.php" method="post">
      <div class="mb-3">
        <label for="email">Email</label><br>
        <input type="email" id="email" class="form-control" name="email" placeholder="...@gmail.com" required>
      </div>

      <div class="mb-3 password-container">
        <label for="password">Password</label><br>
        <input type="password" id="password" class="form-control" name="password" placeholder="*MMbbb1234@.." required>
        <label><input type="checkbox" onclick="togglePassword()"> Show Password</label>
      </div>

      <button class="btn btn-primary w-100" type="submit">Login</button>
      <p class="text-center mt-3"><a href="forgot_password.html">Forgot Password?</a></p>
      <p class="text-center mt-3">Don't have an account? <a href="register.php">Register</a></p>
    </form>
  </div>

  <footer class="footer">
    <div class="social-icons">
      <a href="https://www.facebook.com/profile.php?id=61566355871598" target="_blank"><img src="assets/icons/Facebook icon.jpg" alt="Facebook" width="45" height="45"></a>
      <a href="https://www.instagram.com/_tazania_rugby_development/" target="_blank"><img src="assets/icons/Instagram icon.jpg" alt="Instagram" width="40"></a>
      <a href="https://wa.me/255756624683" target="_blank"><img src="assets/icons/WhatsApp icon.jpg" alt="WhatsApp" width="40"></a>
    </div>
    <div class="footer-logo">
      <img src="assets/icons/trd logo transparent logo.png" alt="TRD Logo" class="logo-transparent" />
    </div>
    <p>&copy; 2025 Tanzania Rugby Development. All rights reserved.</p>
  </footer>

  <script>
    function togglePassword() {
      const input = document.getElementById("password");
      input.type = input.type === "password" ? "text" : "password";
    }
  </script>
</body>
</html>