<?php
session_start();
if (!isset($_SESSION['user'])) {
  header("Location: login.php");
  exit;
}

$role = strtolower($_SESSION['user']['role']);
$email = $_SESSION['user']['email'];
$name = $_SESSION['user']['name'];

$host = "localhost";
$dbname = "trd_donations";
$username = "root";
$password = "";

try {
  $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
  die("Database connection failed: " . $e->getMessage());
}

// Update last_active
$pdo->prepare("UPDATE users SET last_active = NOW() WHERE email = :email")->execute(['email' => $email]);

// Fetch donations
$stmt = $pdo->prepare("SELECT * FROM donations WHERE email = :email ORDER BY timestamp DESC");
$stmt->execute(['email' => $email]);
$myDonations = $stmt->fetchAll(PDO::FETCH_ASSOC);
$hasDonated = count($myDonations) > 0;
$showDashboard = ($role === 'admin' || $hasDonated);

// Total donated
$totalStmt = $pdo->prepare("SELECT SUM(amount) FROM donations WHERE email = :email");
$totalStmt->execute(['email' => $email]);
$totalDonated = $totalStmt->fetchColumn();

// Chart data
$chartData = [];
foreach ($myDonations as $donation) {
  $date = date('Y-m-d', strtotime($donation['timestamp']));
  $chartData[$date] = ($chartData[$date] ?? 0) + $donation['amount'];
}

// Impact stories
$impactStories = [
  ["title" => "Funded jerseys for U15 girls’ team", "quote" => "Now we feel like a real team.", "image" => "u15_girls_team.jpg"],
  ["title" => "Supported travel for TRD youth to Dar", "quote" => "It was our first time seeing the ocean.", "image" => "youth_trip.jpg"],
  ["title" => "Helped host the TRD mini-tournament", "quote" => "We had over 100 kids playing rugby for the first time.", "image" => "mini_tournament.jpg"]
];
$selectedStory = $impactStories[array_rand($impactStories)];

// Profile update
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['update_profile'])) {
  $newName = trim($_POST['name']);
  $newEmail = trim($_POST['email']);
  if ($newName && filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
    $stmt = $pdo->prepare("UPDATE users SET name = :name, email = :email WHERE email = :old_email");
    $stmt->execute(['name' => $newName, 'email' => $newEmail, 'old_email' => $email]);
    $_SESSION['user']['name'] = $newName;
    $_SESSION['user']['email'] = $newEmail;
    header("Location: donor_dashboard.php?updated=1");
    exit;
  }
}

// Feedback submission
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['submit_feedback'])) {
  $feedback = trim($_POST['feedback']);
  if ($feedback) {
    $stmt = $pdo->prepare("INSERT INTO feedback (email, message, submitted_at) VALUES (:email, :message, NOW())");
    $stmt->execute(['email' => $email, 'message' => $feedback]);
  }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Donor Dashboard | TRD</title>
  <link rel="stylesheet" href="style/style.css" />
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    body { font-family: Arial, sans-serif; background: #f4f4f4; margin: 0; padding: 0; color: #000; }
    .header { text-align: center; background: #006400; color: white; padding: 20px; }
    .header .logo { max-height: 70px; margin-bottom: 10px; }
    .nav-links a { color: #FFD700; text-decoration: none; margin: 0 10px; font-weight: bold; }
    .btn { background: linear-gradient(to right, #FFD700, #f0c000); color: #000; padding: 10px 20px; border: none; border-radius: 6px; text-decoration: none; font-weight: bold; box-shadow: 0 2px 5px rgba(0,0,0,0.1); transition: background 0.3s ease; }
    .btn:hover { background: #e6b800; }
    section { padding: 20px; }
    h2, h3 { color: #006400; }
    input, textarea, select { display: block; margin: 10px 0; padding: 8px; width: 100%; max-width: 400px; }
    canvas { max-width: 100%; margin-top: 20px; }
    img { max-width: 100%; border-radius: 8px; margin-top: 10px; }
  </style>
</head>
<body>

<header class="header">
  <div class="header-top">
    <button class="menu-toggle" id="menuToggle">⁝</button>
    <button class="menu-close" id="menuClose">✖</button>
    <img src="assets/icons/trd logo transparent logo.png" alt="TRD Logo" class="logo-transparent" />
    <h1 class="org-name">Tanzania Rugby Development</h1>
  </div>
  <p>Welcome, <?= htmlspecialchars($name) ?>!</p>
  <div class="nav-links">
    <a href="index.html">Home</a>
    <a href="logout.php">Logout</a>
    <?php if ($role === 'admin'): ?>
      <a href="admin.php">Admin Panel</a>
    <?php endif; ?>
  </div>
</header>

<?php if (!$showDashboard): ?>
  <section>
    <h2>Welcome to Your Donor Dashboard</h2>
    <p>Hi <?= htmlspecialchars($name) ?>, we’re excited to have you here!</p>
    <p>Once you make your first donation, this dashboard will show your impact, donation history, and progress.</p>
    <a href="support us.html" class="btn">Support TRD Now</a>
  </section>
<?php else: ?>
  <section>
    <h2>Your Impact</h2>
    <p><strong>Total Donated:</strong> <?= number_format($totalDonated, 2) ?> <?= htmlspecialchars($myDonations[0]['currency'] ?? 'USD') ?></p>

    <h3>Recent Donations</h3>
    <?php foreach ($myDonations as $donation): ?>
      <p>
        <?= number_format($donation['amount'], 2) ?> <?= htmlspecialchars($donation['currency']) ?>
        on <?= date('F j, Y', strtotime($donation['timestamp'])) ?>
        <?= $donation['message'] ? ' — ' . htmlspecialchars($donation['message']) : '' ?>
      </p>
    <?php endforeach; ?>
  </section>

  <section>
    <h2>Your Donation Timeline</h2>
    <canvas id="donationChart" width="400" height="200"></canvas>
    <script>
      const ctx = document.getElementById('donationChart').getContext('2d');
      const chart = new Chart(ctx, {
        type: 'bar',
        data: {
          labels: <?= json_encode(array_keys($chartData)) ?>,
          datasets: [{
            label: 'Amount Donated',
            data: <?= json_encode(array_values($chartData)) ?>,
            backgroundColor: '#FFD700',
            borderColor: '#006400',
            borderWidth: 1
          }]
        },
        options: {
          plugins: { legend: { display: false } },
          scales: {
            y: { beginAtZero: true, title: { display: true, text: 'Amount' } },
            x: { title: { display: true, text: 'Date' } }
          }
        }
      });
    </script>
  </section>

  <section>
    <h2>Impact Story</h2>
    <h3><?= htmlspecialchars($selectedStory['title']) ?></h3>
    <p><em>"<?= htmlspecialchars($selectedStory['quote']) ?>"</em></p>
    <img src="assets/media/<?= htmlspecialchars($selectedStory['image']) ?>" alt="Impact Image" />
  </section>

    <section>
    <h2>Update Your Profile</h2>
    <form method="POST">
      <label for="name">Full Name</label>
      <input type="text" name="name" id="name" value="<?= htmlspecialchars($name) ?>" required />

      <label for="email">Email Address</label>
      <input type="email" name="email" id="email" value="<?= htmlspecialchars($email) ?>" required />

      <button type="submit" name="update_profile" class="btn">Update Profile</button>
    </form>

    <?php if (isset($_GET['updated'])): ?>
      <p style="color:green;">✅ Your profile has been updated successfully.</p>
    <?php endif; ?>
  </section>

  <section>
    <h2>Share Your Thoughts</h2>
    <form method="POST">
      <textarea name="feedback" placeholder="Your feedback or encouragement..." required></textarea>
      <button type="submit" name="submit_feedback" class="btn">Send Feedback</button>
    </form>
  </section>

<?php endif; ?> <!-- ✅ This closes the if (!$showDashboard) block -->
<script> 
// === Mobile Menu Toggle ===
const menuToggle = document.getElementById('menuToggle');
const menuClose = document.getElementById('menuClose');
const navLinks = document.getElementById('navLinks');
const teamsToggle = document.getElementById('teamsToggle');
const teamsMenu = document.getElementById('teamsMenu');

// Open menu
menuToggle.addEventListener('click', () => {
  navLinks.classList.add('show');
  menuToggle.style.display = 'none';
  menuClose.style.display = 'block';
});

// Close menu
menuClose.addEventListener('click', () => {
  navLinks.classList.remove('show');
  menuClose.style.display = 'none';
  menuToggle.style.display = 'block';
  teamsMenu.classList.remove('show'); // Close dropdown too
});

// Auto-collapse menu when a link is clicked (on small screens)
navLinks.querySelectorAll('a').forEach(link => {
  link.addEventListener('click', () => {
    if (window.innerWidth <= 768) {
      navLinks.classList.remove('show');
      menuClose.style.display = 'none';
      menuToggle.style.display = 'block';
      teamsMenu.classList.remove('show');
    }
  });
});

// === Dropdown Toggle for Teams (Click only)
teamsToggle.addEventListener('click', (e) => {
  e.stopPropagation(); // Prevent bubbling
  teamsMenu.classList.toggle('show');
});

// Prevent dropdown from closing when clicking inside
teamsMenu.addEventListener('click', (e) => {
  e.stopPropagation();
});

// Close dropdown if clicking outside
document.addEventListener('click', (e) => {
  if (window.innerWidth <= 768 && !teamsToggle.contains(e.target)) {
    teamsMenu.classList.remove('show');
  }
});

// === Animated Counters ===
const counters = document.querySelectorAll('.counter');

counters.forEach(counter => {
  const updateCount = () => {
    const target = +counter.getAttribute('data-target');
    const count = +counter.innerText;
    const increment = target / 100;

    if (count < target) {
      counter.innerText = Math.ceil(count + increment);
      setTimeout(updateCount, 30);
    } else {
      counter.innerText = target;
    }
  };

  updateCount();
});
</script>
</body>
</html>